<?php
require_once __DIR__.'/config.php';

function make_token(): string {
  // URL-safe random token
  return rtrim(strtr(base64_encode(random_bytes(32)), '+/', '-_'), '=');
}
function hash_token(string $t): string { return hash('sha256', $t); }

function create_session(int $userId, ?int $ttlSec = 2592000 /*30d*/): string {
  $token = make_token();
  $hash  = hash_token($token);
  $exp   = $ttlSec ? date('Y-m-d H:i:s', time()+$ttlSec) : null;

  $c = db();
  $stmt = $c->prepare(
    "INSERT INTO user_sessions (user_id, token_hash, user_agent, ip, expires_at)
     VALUES (?, ?, ?, INET6_ATON(?), ?)"
  );
  $ua = $_SERVER['HTTP_USER_AGENT'] ?? '';
  $ip = $_SERVER['REMOTE_ADDR'] ?? null;
  $stmt->bind_param('issss', $userId, $hash, $ua, $ip, $exp);
  $stmt->execute();

  return $token; // client-কে plaintext token ফেরত দিন (DB তে শুধু hash আছে)
}

function get_bearer_token(): ?string {
  $h = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
  if (stripos($h, 'Bearer ') !== 0) return null;
  $t = trim(substr($h, 7));
  return $t === '' ? null : $t;
}

function auth_user(): ?array {
  $t = get_bearer_token();
  if (!$t) return null;
  $hash = hash_token($t);

  $c = db();
  $sql = "SELECT u.*
          FROM user_sessions s
          JOIN users u ON u.id = s.user_id
          WHERE s.token_hash = ? AND s.revoked = 0
            AND (s.expires_at IS NULL OR s.expires_at > NOW())
          LIMIT 1";
  $stmt = $c->prepare($sql);
  $stmt->bind_param('s', $hash);
  $stmt->execute();
  $res = $stmt->get_result();
  $u   = $res->fetch_assoc();

  return $u ?: null;
}

function require_user(): array {
  $u = auth_user();
  if (!$u) json_fail('NOT_AUTH', 401);
  return $u;
}

function revoke_current_session(): void {
  $t = get_bearer_token();
  if (!$t) return;
  $hash = hash_token($t);
  $c = db();
  $stmt = $c->prepare("UPDATE user_sessions SET revoked=1 WHERE token_hash=?");
  $stmt->bind_param('s', $hash);
  $stmt->execute();
}
