<?php
declare(strict_types=1);

require __DIR__ . '/config.php';
json();

/* ---------- token helpers ---------- */
function make_token(): string {
  return rtrim(strtr(base64_encode(random_bytes(32)), '+/', '-_'), '=');
}
function hash_token(string $t): string { return hash('sha256', $t); }

/* ---------- read body ---------- */
$in   = json_decode(file_get_contents('php://input'), true) ?? $_POST ?? [];
$id   = trim((string)($in['id'] ?? $in['email'] ?? $in['phone'] ?? ''));
$pass = (string)($in['password'] ?? '');

if ($id === '')   { json_fail('MISSING_ID'); }
if ($pass === '') { json_fail('MISSING_PASSWORD'); }

$c = db();

/* ---------- find user by email or phone ---------- */
if (filter_var($id, FILTER_VALIDATE_EMAIL)) {
  $stmt = $c->prepare("SELECT id, uid, email, phone, password_hash, referral_code FROM users WHERE email=? LIMIT 1");
} else {
  $stmt = $c->prepare("SELECT id, uid, email, phone, password_hash, referral_code FROM users WHERE phone=? LIMIT 1");
}
$stmt->bind_param('s', $id);
$stmt->execute();
$res  = $stmt->get_result();
$user = $res ? $res->fetch_assoc() : null;
$stmt->close();

if (!$user) { json_fail('USER_NOT_FOUND'); }

/* ---------- verify password ---------- */
if (!password_verify($pass, (string)$user['password_hash'])) {
  json_fail('WRONG_PASSWORD');
}

/* optional rehash */
if (password_needs_rehash((string)$user['password_hash'], PASSWORD_DEFAULT)) {
  $new = password_hash($pass, PASSWORD_DEFAULT);
  $stmt = $c->prepare("UPDATE users SET password_hash=? WHERE id=? LIMIT 1");
  $stmt->bind_param('si', $new, $user['id']);
  $stmt->execute();
  $stmt->close();
}

/* ---------- issue session token ---------- */
$token     = make_token();
$tokenHash = hash_token($token);
$ua        = $_SERVER['HTTP_USER_AGENT'] ?? '';
$ip        = $_SERVER['REMOTE_ADDR'] ?? null;
$expires   = date('Y-m-d H:i:s', time() + 30*24*60*60); // 30 days

$stmt = $c->prepare("
  INSERT INTO user_sessions (user_id, token_hash, user_agent, ip, expires_at)
  VALUES (?, ?, ?, INET6_ATON(?), ?)
");
$stmt->bind_param('issss', $user['id'], $tokenHash, $ua, $ip, $expires);
$stmt->execute();
$stmt->close();

/* ---------- OPTIONAL last_login_at update; ignore if column missing ---------- */
try {
  $stmt = $c->prepare("UPDATE users SET last_login_at=NOW() WHERE id=? LIMIT 1");
  $stmt->bind_param('i', $user['id']);
  $stmt->execute();
  $stmt->close();
} catch (Throwable $e) {
  // Column নেই হলে/অন্য কোনো কারণে fail হলে চুপচাপ ইগনোর
}

/* ---------- success JSON ---------- */
json_ok([
  'token' => $token,  // plaintext token -> client stores securely
  'user'  => [
    'uid'          => (string)$user['uid'],
    'email'        => (string)$user['email'],
    'phone'        => (string)($user['phone'] ?? ''),
    'referralCode' => (string)$user['referral_code'],
  ],
]);
