<?php
// Quantum Exchange - Profile API (FINAL)
// Returns: { ok: true, data: {...} }  OR  { ok: false, error: "..." }

require_once __DIR__ . '/config.php';

try {
  $conn = db();

  // ---------- 1) Read token/header/query safely ----------
  $auth = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
  $xAuth = $_SERVER['HTTP_X_AUTH_TOKEN'] ?? '';
  $getToken = $_GET['token'] ?? '';
  $token = '';

  // Authorization: Bearer <token>
  if ($auth) {
    if (stripos($auth, 'Bearer ') === 0) $token = trim(substr($auth, 7));
    else $token = trim($auth);
  } elseif ($xAuth) {
    $token = trim($xAuth);
  } elseif ($getToken) {
    $token = trim($getToken);
  }

  $uid = trim($_GET['uid'] ?? '');
  $ref = trim($_GET['ref'] ?? '');

  // ---------- 2) Try to locate the user ----------
  // NOTE: we keep it robust for your DB — first try sessions join (if exists),
  // then fallback to users.token, finally allow uid/ref lookup (for dev).

  $user = null;

  // 2.a) sessions join if table exists
  $hasSessions = false;
  $rs = $conn->query("SHOW TABLES LIKE 'sessions'");
  if ($rs && $rs->num_rows === 1) $hasSessions = true;
  if ($rs) $rs->close();

  if ($token !== '' && $hasSessions) {
    try {
      $q = $conn->prepare("
        SELECT u.* FROM sessions s
        JOIN users u ON u.id = s.user_id
        WHERE s.token = ?
        LIMIT 1
      ");
      $q->bind_param('s', $token);
      $q->execute();
      $res = $q->get_result();
      if ($res && $res->num_rows === 1) {
        $user = $res->fetch_assoc();
      }
      $q->close();
    } catch (Throwable $e) {
      // ignore, fallback below
    }
  }

  // 2.b) fallback: users.token
  if (!$user && $token !== '') {
    try {
      $q = $conn->prepare("SELECT * FROM users WHERE token = ? LIMIT 1");
      $q->bind_param('s', $token);
      $q->execute();
      $res = $q->get_result();
      if ($res && $res->num_rows === 1) {
        $user = $res->fetch_assoc();
      }
      $q->close();
    } catch (Throwable $e) {
      // ignore
    }
  }

  // 2.c) dev fallback: uid or referral code (for web debug only)
  if (!$user && ($uid !== '' || $ref !== '')) {
    if ($uid !== '') {
      $q = $conn->prepare("SELECT * FROM users WHERE uid = ? LIMIT 1");
      $q->bind_param('s', $uid);
      $q->execute();
      $res = $q->get_result();
      if ($res && $res->num_rows === 1) $user = $res->fetch_assoc();
      $q->close();
    }
    if (!$user && $ref !== '') {
      $q = $conn->prepare("SELECT * FROM users WHERE referral_code = ? LIMIT 1");
      $q->bind_param('s', $ref);
      $q->execute();
      $res = $q->get_result();
      if ($res && $res->num_rows === 1) $user = $res->fetch_assoc();
      $q->close();
    }
  }

  if (!$user) {
    // Always a JSON with ok:false so client never sees "Invalid response"
    json_fail('NOT_FOUND', 200);
  }

  // ---------- 3) Build response payload safely ----------
  $email = (string)($user['email'] ?? '');
  $uidOut = (string)($user['uid'] ?? ($user['id'] ?? ''));
  $refOut = (string)($user['referral_code'] ?? ($user['ref'] ?? ''));
  $vip = (int)($user['vip_level'] ?? 0);
  $kycLevel = (int)($user['kyc_level'] ?? 0);
  $isVerified = (int)($user['is_verified'] ?? 0) > 0 || $kycLevel > 0;

  // Email mask: al**@**.com style
  $masked = $email;
  if (strpos($email, '@') !== false) {
    [$local, $domain] = explode('@', $email, 2);
    $localMask = strlen($local) > 2 ? substr($local, 0, 2) . '**' : $local . '*';
    // hide domain middle part too
    $domainMask = '**@**';
    $masked = $localMask . '@' . $domain;
    // match your desired "al**@**.com" look
    if (($dot = strrpos($domain, '.')) !== false) {
      $masked = $localMask . '@**' . substr($domain, $dot);
    }
  }

  $data = [
    'email'           => $email,
    'email_masked'    => $masked,
    'uid'             => $uidOut,
    'referral_code'   => $refOut,
    'vip_level'       => $vip,              // e.g., 0 => "LV.0"
    'kyc_level'       => $kycLevel,         // 0 => "Unverified"
    'safeguard'       => true,              // static flag for now
    'verified'        => $isVerified,
    'commission_rate' => 0.15,              // 15% default (ref page)
  ];

  json_ok($data);

} catch (Throwable $e) {
  // Never leak PHP errors to client as HTML; always JSON
  json_fail('SERVER_ERROR', 200, $e->getMessage());
}
