<?php
require_once __DIR__ . '/../config/db.php';
require_once __DIR__ . '/../config/helpers.php';
require_once __DIR__ . '/../lib/crypto.php';
require_once __DIR__ . '/../lib/bybit_client.php';

function trade_resolve_user_id(PDO $pdo, array $input): int {
    if (isset($input['user_id']) && is_numeric($input['user_id'])) {
        return (int)$input['user_id'];
    }
    if (isset($input['email'])) {
        $email = trim((string)$input['email']);
        if ($email !== '') {
            $st = $pdo->prepare('SELECT id FROM users WHERE email = :e LIMIT 1');
            $st->execute([':e' => $email]);
            $row = $st->fetch();
            if ($row && isset($row['id'])) return (int)$row['id'];
        }
    }
    json_response(['success'=>false,'message'=>'user_id or email required'], 400);
}

function trade_get_setting(PDO $pdo, string $key, string $default=''): string {
    $st = $pdo->prepare('SELECT setting_value FROM app_settings WHERE setting_key = :k LIMIT 1');
    $st->execute([':k' => $key]);
    $row = $st->fetch();
    if ($row && isset($row['setting_value'])) return (string)$row['setting_value'];
    return $default;
}

function trade_ensure_access_row(PDO $pdo, int $userId): array {
    $st = $pdo->prepare('SELECT * FROM user_trade_access WHERE user_id = :uid LIMIT 1');
    $st->execute([':uid' => $userId]);
    $row = $st->fetch();
    if ($row) return $row;

    $ins = $pdo->prepare('INSERT INTO user_trade_access (user_id, api_status, memo_verified, trade_unlocked) VALUES (:uid, "none", 0, 0)');
    $ins->execute([':uid' => $userId]);

    $st2 = $pdo->prepare('SELECT * FROM user_trade_access WHERE user_id = :uid LIMIT 1');
    $st2->execute([':uid' => $userId]);
    return $st2->fetch() ?: [];
}

function trade_get_user_keys(PDO $pdo, int $userId): array {
    $row = trade_ensure_access_row($pdo, $userId);

    $apiKeyEnc = $row['api_key_enc'] ?? '';
    $apiSecEnc = $row['api_secret_enc'] ?? '';

    $apiKey = decrypt_text((string)$apiKeyEnc);
    $apiSecret = decrypt_text((string)$apiSecEnc);

    return [
        'api_status' => (string)($row['api_status'] ?? 'none'),
        'trade_unlocked' => (int)($row['trade_unlocked'] ?? 0),
        'memo_verified' => (int)($row['memo_verified'] ?? 0),
        'api_key' => $apiKey,
        'api_secret' => $apiSecret,
    ];
}

function trade_require_api_approved(PDO $pdo, int $userId): array {
    $k = trade_get_user_keys($pdo, $userId);

    if ((int)$k['trade_unlocked'] !== 1) {
        json_response(['success'=>false,'message'=>'Trade locked'], 403);
    }

    if (strtolower((string)$k['api_status']) !== 'approved') {
        json_response(['success'=>false,'message'=>'API not approved'], 403);
    }

    if (trim((string)$k['api_key']) === '' || trim((string)$k['api_secret']) === '') {
        json_response(['success'=>false,'message'=>'Missing API credentials'], 403);
    }

    return $k;
}

function bybit_pick_usdt_available_balance(array $walletBalanceJson): float {
    // Try to extract USDT available balance from Bybit response.
    // Response structure varies; we attempt common fields.
    // We do NOT throw here; return 0 when unknown.

    if (!isset($walletBalanceJson['result']['list']) || !is_array($walletBalanceJson['result']['list'])) {
        return 0.0;
    }

    foreach ($walletBalanceJson['result']['list'] as $acc) {
        if (!isset($acc['coin']) || !is_array($acc['coin'])) continue;
        foreach ($acc['coin'] as $coin) {
            $c = strtoupper((string)($coin['coin'] ?? ''));
            if ($c !== 'USDT') continue;

            // Prefer availableToWithdraw / availableBalance / walletBalance
            foreach (['availableToWithdraw','availableBalance','walletBalance'] as $k) {
                if (isset($coin[$k]) && is_numeric($coin[$k])) return (float)$coin[$k];
                if (isset($coin[$k]) && is_string($coin[$k]) && is_numeric($coin[$k])) return (float)$coin[$k];
            }
        }
    }

    return 0.0;
}
