<?php
// api/user/assets_summary.php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type');

require_once __DIR__ . '/../config.php';

function send_json($data) {
    echo json_encode($data);
    exit;
}

/**
 * Bybit থেকে যে কোন symbol এর USDT price আনে।
 * BTC -> BTCUSDT, ETH -> ETHUSDT ইত্যাদি।
 * USDT/USড এর case এ 1.0 ফেরত দেয়।
 */
function fetch_usdt_price_for(string $baseSymbol): float {
    static $cache = [];

    $baseSymbol = strtoupper(trim($baseSymbol));
    if ($baseSymbol === '') {
        return 0.0;
    }

    if (isset($cache[$baseSymbol])) {
        return $cache[$baseSymbol];
    }

    if ($baseSymbol === 'USDT' || $baseSymbol === 'USD') {
        $cache[$baseSymbol] = 1.0;
        return 1.0;
    }

    $pair = $baseSymbol . 'USDT';
    $url  = 'https://api.bybit.com/v5/market/tickers?category=linear&symbol=' . urlencode($pair);

    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT        => 5,
    ]);

    $res  = curl_exec($ch);
    if ($res === false) {
        curl_close($ch);
        $cache[$baseSymbol] = 0.0;
        return 0.0;
    }

    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($code !== 200) {
        $cache[$baseSymbol] = 0.0;
        return 0.0;
    }

    $data = json_decode($res, true);
    if (!isset($data['retCode']) || $data['retCode'] != 0) {
        $cache[$baseSymbol] = 0.0;
        return 0.0;
    }

    if (
        !isset($data['result']['list'][0]['lastPrice']) ||
        !is_numeric($data['result']['list'][0]['lastPrice'])
    ) {
        $cache[$baseSymbol] = 0.0;
        return 0.0;
    }

    $price = (float)$data['result']['list'][0]['lastPrice'];
    $cache[$baseSymbol] = $price;

    return $price;
}

try {
    $db = getDb();
} catch (Throwable $e) {
    send_json([
        'success' => false,
        'message' => 'Database connection failed',
        'error'   => $e->getMessage(),
    ]);
}

// -------- 1) ইনপুট থেকে ইমেইল ----------
$input = json_decode(file_get_contents('php://input'), true);
$email = isset($input['email']) ? trim($input['email']) : '';

if ($email === '') {
    send_json([
        'success' => false,
        'message' => 'Email required',
    ]);
}

// -------- 2) ইউজার ----------
$stmt = $db->prepare('SELECT id, name, email FROM users WHERE email = ? LIMIT 1');
$stmt->execute([$email]);
$user = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$user) {
    send_json([
        'success' => false,
        'message' => 'User not found',
    ]);
}

$userId = (int)$user['id'];

// -------- 3) user_assets থেকে holdings ----------
$stmt = $db->prepare('SELECT symbol, name, amount, usd_value, pnl_usd, pnl_percent 
                      FROM user_assets 
                      WHERE user_id = ?
                      ORDER BY usd_value DESC');
$stmt->execute([$userId]);

$assets      = [];
$totalUsd    = 0.0;
$totalPnlUsd = 0.0;

$updateStmt = $db->prepare(
    'UPDATE user_assets SET usd_value = ? WHERE user_id = ? AND symbol = ?'
);

while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
    $symbol = strtoupper($row['symbol'] ?? '');
    $name   = $row['name'] ?? $symbol;
    $amount = (float)($row['amount'] ?? 0);
    $oldUsd = (float)($row['usd_value'] ?? 0);
    $pnlUsd = (float)($row['pnl_usd'] ?? 0);
    $pnlPct = (float)($row['pnl_percent'] ?? 0);

    $price = fetch_usdt_price_for($symbol);

    if ($price > 0) {
        $usdValue = $amount * $price;

        // DB-তে নতুন usd_value সেভ করি (fallback হিসেবে কাজে লাগবে)
        $updateStmt->execute([$usdValue, $userId, $symbol]);
    } else {
        // Bybit fail করলে আগের stored usd_value ব্যবহার করি
        $usdValue = $oldUsd;
    }

    $totalUsd    += $usdValue;
    $totalPnlUsd += $pnlUsd;

    $assets[] = [
        'symbol'      => $symbol,
        'name'        => $name,
        'amount'      => $amount,
        'usd_value'   => $usdValue,
        'pnl_usd'     => $pnlUsd,
        'pnl_percent' => $pnlPct,
    ];
}

// -------- 4) funding / unified wallet আপডেট ----------
$fundingUsd = $totalUsd;
$unifiedUsd = 0.0;

// funding wallet
$stmt = $db->prepare(
    'SELECT id, balance_usd FROM user_wallets WHERE user_id = ? AND wallet_type = ? LIMIT 1'
);
$stmt->execute([$userId, 'funding']);
$fund = $stmt->fetch(PDO::FETCH_ASSOC);

if ($fund) {
    $upd = $db->prepare('UPDATE user_wallets SET balance_usd = ? WHERE id = ?');
    $upd->execute([$fundingUsd, $fund['id']]);
} else {
    $ins = $db->prepare(
        'INSERT INTO user_wallets (user_id, wallet_type, balance_usd) VALUES (?, ?, ?)'
    );
    $ins->execute([$userId, 'funding', $fundingUsd]);
}

// unified wallet
$stmt->execute([$userId, 'unified']);
$uni = $stmt->fetch(PDO::FETCH_ASSOC);

if ($uni) {
    $upd = $db->prepare('UPDATE user_wallets SET balance_usd = ? WHERE id = ?');
    $upd->execute([$unifiedUsd, $uni['id']]);
} else {
    $ins = $db->prepare(
        'INSERT INTO user_wallets (user_id, wallet_type, balance_usd) VALUES (?, ?, ?)'
    );
    $ins->execute([$userId, 'unified', $unifiedUsd]);
}

// users.balance_usd = totalUsd
$updUser = $db->prepare('UPDATE users SET balance_usd = ? WHERE id = ?');
$updUser->execute([$totalUsd, $userId]);

// -------- 5) BTC-equivalent + PnL ----------
$btcPrice = fetch_usdt_price_for('BTC');
$btcEq    = $btcPrice > 0 ? $totalUsd / $btcPrice : 0.0;

$pnlUsd = $totalPnlUsd;
$pnlPct = $totalUsd > 0 ? ($pnlUsd / $totalUsd) * 100.0 : 0.0;

// -------- 6) Response ----------
send_json([
    'success' => true,
    'data'    => [
        'total_usd'      => $totalUsd,
        'funding_usd'    => $fundingUsd,
        'unified_usd'    => $unifiedUsd,
        'btc_equivalent' => $btcEq,
        'pnl_usd'        => $pnlUsd,
        'pnl_percent'    => $pnlPct,
        'assets'         => $assets,
    ],
]);
