<?php
// api/user/profile.php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type');

require_once __DIR__ . '/../config.php';

function send_json($data) {
    echo json_encode($data);
    exit;
}

// price helper – assets_summary.php এর মতই
function fetch_usdt_price_for(string $baseSymbol): float {
    static $cache = [];

    $baseSymbol = strtoupper(trim($baseSymbol));
    if ($baseSymbol === '') return 0.0;

    if (isset($cache[$baseSymbol])) return $cache[$baseSymbol];

    if ($baseSymbol === 'USDT' || $baseSymbol === 'USD') {
        return $cache[$baseSymbol] = 1.0;
    }

    $pair = $baseSymbol . 'USDT';
    $url  = 'https://api.bybit.com/v5/market/tickers?category=linear&symbol=' . urlencode($pair);

    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT        => 5,
    ]);

    $res = curl_exec($ch);
    if ($res === false) {
        curl_close($ch);
        return $cache[$baseSymbol] = 0.0;
    }

    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($code !== 200) {
        return $cache[$baseSymbol] = 0.0;
    }

    $data = json_decode($res, true);
    if (!isset($data['retCode']) || $data['retCode'] != 0) {
        return $cache[$baseSymbol] = 0.0;
    }

    if (
        !isset($data['result']['list'][0]['lastPrice']) ||
        !is_numeric($data['result']['list'][0]['lastPrice'])
    ) {
        return $cache[$baseSymbol] = 0.0;
    }

    $price = (float)$data['result']['list'][0]['lastPrice'];
    return $cache[$baseSymbol] = $price;
}

try {
    $db = getDb();
} catch (Throwable $e) {
    send_json([
        'success' => false,
        'message' => 'Database connection failed',
        'error'   => $e->getMessage(),
    ]);
}

// ইনপুট
$input = json_decode(file_get_contents('php://input'), true);
$email = isset($input['email']) ? trim($input['email']) : '';

if ($email === '') {
    send_json([
        'success' => false,
        'message' => 'Email required',
    ]);
}

// user
$stmt = $db->prepare('SELECT id, name, email, role FROM users WHERE email = ? LIMIT 1');
$stmt->execute([$email]);
$user = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$user) {
    send_json([
        'success' => false,
        'message' => 'User not found',
    ]);
}

$userId = (int)$user['id'];

// assets থেকে total বের করি (assets_summary এর মত)
$stmt = $db->prepare('SELECT symbol, amount, usd_value, pnl_usd FROM user_assets WHERE user_id = ?');
$stmt->execute([$userId]);

$totalUsd    = 0.0;
$totalPnlUsd = 0.0;

$updateStmt = $db->prepare(
    'UPDATE user_assets SET usd_value = ? WHERE user_id = ? AND symbol = ?'
);

while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
    $symbol = strtoupper($row['symbol'] ?? '');
    $amount = (float)($row['amount'] ?? 0);
    $oldUsd = (float)($row['usd_value'] ?? 0);
    $pnlUsd = (float)($row['pnl_usd'] ?? 0);

    $price = fetch_usdt_price_for($symbol);

    if ($price > 0) {
        $usdValue = $amount * $price;
        $updateStmt->execute([$usdValue, $userId, $symbol]);
    } else {
        $usdValue = $oldUsd;
    }

    $totalUsd    += $usdValue;
    $totalPnlUsd += $pnlUsd;
}

// users.balance_usd update
$upd = $db->prepare('UPDATE users SET balance_usd = ? WHERE id = ?');
$upd->execute([$totalUsd, $userId]);

$btcPrice = fetch_usdt_price_for('BTC');
$btcEq    = $btcPrice > 0 ? $totalUsd / $btcPrice : 0.0;

$pnlUsd = $totalPnlUsd;
$pnlPct = $totalUsd > 0 ? ($pnlUsd / $totalUsd) * 100.0 : 0.0;

send_json([
    'success' => true,
    'data'    => [
        'name'           => $user['name'],
        'email'          => $user['email'],
        'role'           => $user['role'],
        'total_usd'      => $totalUsd,
        'balance_usd'    => $totalUsd,     // পুরনো ফ্রন্টএন্ডের সাথে compatible রাখলাম
        'btc_equivalent' => $btcEq,
        'pnl_usd'        => $pnlUsd,
        'pnl_percent'    => $pnlPct,
    ],
]);
