<?php
// api/user/wallet_transfer.php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type');

require_once __DIR__ . '/../config.php';

function send_json($arr) {
    echo json_encode($arr);
    exit;
}

try {
    $db = getDb();
} catch (Throwable $e) {
    send_json([
        'success' => false,
        'message' => 'Database connection failed',
        'error'   => $e->getMessage(),
    ]);
}

$input = json_decode(file_get_contents('php://input'), true);

$email      = isset($input['email']) ? trim($input['email']) : '';
$fromWallet = isset($input['from']) ? trim($input['from']) : '';
$toWallet   = isset($input['to']) ? trim($input['to']) : '';
$amount     = isset($input['amount']) ? (float)$input['amount'] : 0.0;

$validTypes = ['funding', 'unified'];

if ($email === '' || !in_array($fromWallet, $validTypes, true) || !in_array($toWallet, $validTypes, true)) {
    send_json([
        'success' => false,
        'message' => 'Invalid parameters',
    ]);
}
if ($fromWallet === $toWallet) {
    send_json([
        'success' => false,
        'message' => 'From and to wallet must be different',
    ]);
}
if ($amount <= 0) {
    send_json([
        'success' => false,
        'message' => 'Amount must be positive',
    ]);
}

$stmt = $db->prepare('SELECT id, balance_usd FROM users WHERE email = ? LIMIT 1');
$stmt->execute([$email]);
$user = $stmt->fetch(PDO::FETCH_ASSOC);
if (!$user) {
    send_json([
        'success' => false,
        'message' => 'User not found',
    ]);
}
$userId = (int)$user['id'];

try {
    $db->beginTransaction();

    // দুই wallet lock করে নেই
    $q = $db->prepare('SELECT wallet_type, balance_usd FROM user_wallets WHERE user_id = ? FOR UPDATE');
    $q->execute([$userId]);
    $rows = $q->fetchAll(PDO::FETCH_ASSOC);

    $balances = [
        'funding' => 0.0,
        'unified' => 0.0,
    ];

    foreach ($rows as $r) {
        $type = $r['wallet_type'];
        if (isset($balances[$type])) {
            $balances[$type] = (float)$r['balance_usd'];
        }
    }

    // ensure exists
    foreach ($validTypes as $t) {
        if (!in_array($t, array_column($rows, 'wallet_type'), true)) {
            $ins = $db->prepare('INSERT INTO user_wallets (user_id, wallet_type, balance_usd) VALUES (?, ?, 0)');
            $ins->execute([$userId, $t]);
        }
    }

    if ($balances[$fromWallet] < $amount) {
        $db->rollBack();
        send_json([
            'success' => false,
            'message' => 'Insufficient balance in ' . $fromWallet,
        ]);
    }

    $balances[$fromWallet] -= $amount;
    $balances[$toWallet]   += $amount;

    $upd = $db->prepare('UPDATE user_wallets SET balance_usd = ? WHERE user_id = ? AND wallet_type = ?');
    $upd->execute([$balances['funding'], $userId, 'funding']);
    $upd->execute([$balances['unified'], $userId, 'unified']);

    // users.balance_usd = funding + unified
    $total = $balances['funding'] + $balances['unified'];
    $upd2 = $db->prepare('UPDATE users SET balance_usd = ? WHERE id = ?');
    $upd2->execute([$total, $userId]);

    $db->commit();

    send_json([
        'success' => true,
        'message' => 'Transfer successful',
        'balances' => [
            'funding_usd' => $balances['funding'],
            'unified_usd' => $balances['unified'],
            'total_usd'   => $total,
        ],
    ]);
} catch (Throwable $e) {
    if ($db->inTransaction()) {
        $db->rollBack();
    }
    send_json([
        'success' => false,
        'message' => 'Transfer failed',
        'error'   => $e->getMessage(),
    ]);
}
